var __extends = this.__extends || function (d, b) {
    function __() { this.constructor = d; }
    __.prototype = b.prototype;
    d.prototype = new __();
};
var VisualProfiler;
(function (VisualProfiler) {
    (function (Graph) {
        "use strict";
        (function (AxisPositionType) {
            AxisPositionType._map = [];
            AxisPositionType.top = 0;
            AxisPositionType.middle = 1;
            AxisPositionType.bottom = 2;
        })(Graph.AxisPositionType || (Graph.AxisPositionType = {}));
        var AxisPositionType = Graph.AxisPositionType;
        (function (ColorSchemeType) {
            ColorSchemeType._map = [];
            ColorSchemeType.VizSurfaceGrayDark = 0;
            ColorSchemeType.VizSurfaceStrongBlueDark = 1;
            ColorSchemeType.VizSurfaceDarkGoldDark = 2;
            ColorSchemeType.VizSurfaceChartreuseGreenDark = 3;
        })(Graph.ColorSchemeType || (Graph.ColorSchemeType = {}));
        var ColorSchemeType = Graph.ColorSchemeType;
        (function (ScaleType) {
            ScaleType._map = [];
            ScaleType.Left = 0;
            ScaleType.Right = 1;
        })(Graph.ScaleType || (Graph.ScaleType = {}));
        var ScaleType = Graph.ScaleType;
        var Point2d = (function () {
            function Point2d(x, y, data, isInView, seriesIndex, xTranslate, yTranslate, xPx, yPx) {
                if (typeof xTranslate === "undefined") { xTranslate = 0; }
                if (typeof yTranslate === "undefined") { yTranslate = -1; }
                this.x = x;
                this.y = y;
                this.data = data;
                this.isInView = isInView;
                this.seriesIndex = seriesIndex;
                this.xTranslate = xTranslate;
                this.yTranslate = yTranslate;
                this.xPx = xPx;
                this.yPx = yPx;
            }
            return Point2d;
        })();
        Graph.Point2d = Point2d;        
        var Padding = (function () {
            function Padding(left, top, right, bottom) {
                this.left = left;
                this.top = top;
                this.right = right;
                this.bottom = bottom;
            }
            return Padding;
        })();
        Graph.Padding = Padding;        
        var RectangleDimension = (function (_super) {
            __extends(RectangleDimension, _super);
            function RectangleDimension(left, top, right, bottom) {
                        _super.call(this, left, top, right, bottom);
                if(this.left > this.right || this.top > this.bottom) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1054"));
                }
            }
            Object.defineProperty(RectangleDimension.prototype, "width", {
                get: function () {
                    return this.right - this.left;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(RectangleDimension.prototype, "height", {
                get: function () {
                    return this.bottom - this.top;
                },
                enumerable: true,
                configurable: true
            });
            return RectangleDimension;
        })(Padding);
        Graph.RectangleDimension = RectangleDimension;        
        var MinMax = (function () {
            function MinMax(min, max) {
                this.min = min;
                this.max = max;
            }
            Object.defineProperty(MinMax.prototype, "range", {
                get: function () {
                    return this.max - this.min;
                },
                enumerable: true,
                configurable: true
            });
            return MinMax;
        })();
        Graph.MinMax = MinMax;        
        var ChartColorScheme = (function () {
            function ChartColorScheme(lineColorString, lineFillColorString) {
                this._lineColorString = "#FF0000";
                this._lineFillColorString = "#FF0000";
                this._lineColorString = lineColorString;
                this._lineFillColorString = lineFillColorString;
            }
            Object.defineProperty(ChartColorScheme.prototype, "lineColor", {
                get: function () {
                    return this._lineColorString;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(ChartColorScheme.prototype, "lineFillColor", {
                get: function () {
                    return this._lineFillColorString;
                },
                enumerable: true,
                configurable: true
            });
            return ChartColorScheme;
        })();
        Graph.ChartColorScheme = ChartColorScheme;        
        var GraphConfigurations = (function () {
            function GraphConfigurations() {
                this.chartAxisPadding = new Padding(0, 0, 0, 0);
                this.chartAxisCount = 5;
                this.chartAxisIncreaseRatio = 1.1;
                this.scaleForWhiteSpace = 0.05;
                this.chartDrawFill = true;
                this.gridX = new MinMax(0, 0);
                this.gridY = new MinMax(0, 100);
                this.axes = [];
                this.colorSchemeType = ColorSchemeType.VizSurfaceChartreuseGreenDark;
                this.legendContainerId = "legendDiv";
                this.graphHeight = 88;
                this.headerId = "headerContainer";
                this.header = {
                    containerId: "headerContainer",
                    titleConfig: {
                        containerId: "titleContainer",
                        text: "Visual Throughput",
                        isExpanded: true,
                        unit: "FPS"
                    },
                    legendConfig: {
                        containerId: "legendContainer",
                        data: []
                    }
                };
                this.graphScaleWidthLeft = 40;
                this.graphScaleWidthRight = 40;
                this.leftScaleBorder = 1;
                this.rightScaleBorder = 1;
                this.pixelRange = 5;
                this.graphContainerId = "graphBody";
            }
            return GraphConfigurations;
        })();
        Graph.GraphConfigurations = GraphConfigurations;        
        var AnimationFrameGraph = (function () {
            function AnimationFrameGraph(config) {
                this._series = [];
                this._selectionStartXPercent = 0;
                this._selectionWidthPercent = 100;
                this._graphContainerCss = "graphContainer";
                this._isExpanded = true;
                this._tabFocusPointList = [];
                this._graphConfiguration = new GraphConfigurations();
                this._container = document.getElementById(config.containerId);
                this._graphConfiguration.containerId = config.containerId;
                if(this._container === null) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
                this._container.className = this._graphContainerCss;
                this._config = config;
                this.initialize();
                while(this._container.childNodes.length > 0) {
                    this._container.removeChild(this._container.firstChild);
                }
                this._container.style.backgroundColor = Plugin.Theme.getValue("vs-system-window-color");
                this._header = document.createElement("div");
                this._header.className = "graph-header";
                this._header.id = this._graphConfiguration.header.containerId;
                this._container.appendChild(this._header);
                this._body = document.createElement("div");
                this._body.className = "graph-body";
                this._body.id = this._graphConfiguration.graphContainerId;
                this._body.style.height = this._graphConfiguration.graphHeight + "px";
                this._container.appendChild(this._body);
                this._graphScaleLeft = new Scale(ScaleType.Left, this._graphConfiguration);
                this._graphScaleRight = new Scale(ScaleType.Right, this._graphConfiguration);
                this._canvasDiv = document.createElement("div");
                this._canvasDiv.className = "graph-canvas-div";
                this._canvasDiv.tabIndex = 0;
                this._canvasDiv.setAttribute("aria-label", Plugin.Resources.getString("VisualThroughputGraphAriaLabel"));
                if(this._keyPress) {
                    this._canvasDiv.addEventListener("keydown", this._keyPress, true);
                }
                this._body.appendChild(this._canvasDiv);
                this._canvas = document.createElement("canvas");
                this._canvasDiv.appendChild(this._canvas);
                this._canvas.className = "graph-canvas";
                this._canvas.style.background = Plugin.Theme.getValue("vs-system-window-color");
                this._canvas.addEventListener('mousemove', this.onMouseOver.bind(this));
                this._canvas.addEventListener('mouseout', this.onMouseOut.bind(this));
                if(!this._container.runtimeStyle.position || this._container.runtimeStyle.position === "static") {
                    this._container.style.position = "relative";
                }
                this._context = this._canvas.getContext("2d");
                for(var i = 0; i < this._config.series.length; i++) {
                    this.addSeries(this._config.series[i].legend, this._config.series[i].counterId, this._config.series[i].colorSchemeType);
                    var color = this._series[this._series.length - 1].colorScheme.lineColor;
                    this._graphConfiguration.header.legendConfig.data.push({
                        color: color,
                        text: this._config.series[i].legend
                    });
                }
                this._axes = new Axes(this._graphConfiguration);
                var header = new Header(this._graphConfiguration.header);
                header.title.registerCallback(this.toggleDisplay.bind(this));
                header.render();
                window.addEventListener("resize", this.onResize.bind(this));
                this.calculateChartSize();
                this._canvasDiv.addEventListener("mousedown", this.startSelection.bind(this));
                this._dragHandler = this.drag.bind(this);
                Plugin.Theme.addEventListener("themechanged", this.onThemeChanged.bind(this));
            }
            AnimationFrameGraph._cursor = null;
            Object.defineProperty(AnimationFrameGraph.prototype, "selection", {
                set: function (selection) {
                    if(!selection) {
                        return;
                    }
                    var startXPercent = (selection.begin.nsec - this._config.xRange.min) / this.xWidth * 100;
                    var widthPercent = selection.elapsed.nsec / this.xWidth * 100;
                    this._selectionStartXPercent = startXPercent;
                    this._selectionWidthPercent = widthPercent;
                    this.removeFocus();
                    this.drawSelection();
                    this.createPointIndexInfoList();
                },
                enumerable: true,
                configurable: true
            });
            AnimationFrameGraph.prototype.render = function () {
                if(this._graphConfiguration.chartAxisPadding.left <= this._graphConfiguration.chartWidth && this._graphConfiguration.chartAxisPadding.top <= this._graphConfiguration.chartHeight) {
                    this.drawChartBackground();
                    this._graphScaleLeft.render();
                    this._graphScaleRight.render();
                    this._graphConfiguration.chartRect = new RectangleDimension(this._graphConfiguration.chartAxisPadding.left, this._graphConfiguration.chartAxisPadding.top, this._graphConfiguration.chartWidth, this._graphConfiguration.chartHeight);
                    for(var seriesIndex = 0; seriesIndex < this._series.length; seriesIndex++) {
                        var series = this._series[seriesIndex];
                        series.render();
                    }
                    this._context.save();
                    this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                    this._context.fillRect(0, this._graphConfiguration.chartRect.top, this._graphConfiguration.chartAxisPadding.left, this._graphConfiguration.chartRect.height);
                    this._context.restore();
                    if(this._axes) {
                        this._axes.render();
                    }
                }
                this.addVerticalRulerLines();
                this.createPointIndexInfoList();
            };
            AnimationFrameGraph.prototype.setViewportRange = function (range) {
                if(this._graphConfiguration.gridX.max !== range.max || this._graphConfiguration.gridX.min !== range.min) {
                    this.calculateChartSize();
                    this._graphConfiguration.gridX.min = range.min;
                    this._graphConfiguration.gridX.max = range.max;
                }
                this.removeFocus();
                this.render();
                this._dragDirection = VisualProfiler.DragDirection.none;
                this.clearSelection();
            };
            AnimationFrameGraph.prototype.setSeriesDataSource = function (counterId, points) {
                var seriesData;
                for(var i = 0; i < this._series.length; i++) {
                    if(this._series[i].counterId === counterId) {
                        this._series[i].data = points ? points : [];
                        for(var j = 0; j < this._series[i].data.length; j++) {
                            var point = this._series[i].data[j];
                            if(typeof point.data === "undefined" || point.data === null) {
                                point.data = point.y.toFixed(0);
                            }
                        }
                    }
                }
                this.calculateYRange(this.getMaxPoint(points));
            };
            Object.defineProperty(AnimationFrameGraph.prototype, "xWidth", {
                get: function () {
                    return this._config.xRange.max - this._config.xRange.min;
                },
                enumerable: true,
                configurable: true
            });
            AnimationFrameGraph.prototype.toggleDisplay = function (visible) {
                this._body.style.display = visible ? "-ms-grid" : "none";
                this._isExpanded = visible;
                if(this._isExpanded && this.calculateChartSize()) {
                    this.render();
                }
                if(visible) {
                    Plugin.VS.Internal.CodeMarkers.fire(VisualProfiler.CodeMarkerValues.perfBrowserTools_VisualProfilerGraphExpanded);
                } else {
                    Plugin.VS.Internal.CodeMarkers.fire(VisualProfiler.CodeMarkerValues.perfBrowserTools_VisualProfilerGraphCollapsed);
                }
            };
            AnimationFrameGraph.prototype.onThemeChanged = function (args) {
                this._container.style.backgroundColor = Plugin.Theme.getValue("vs-system-window-color");
                this.render();
            };
            AnimationFrameGraph.prototype.initialize = function () {
                if(!this._config.axes || this._config.axes.length === 0) {
                    this._graphConfiguration.axes = [
                        {
                            value: 30
                        }, 
                        {
                            value: 60
                        }
                    ];
                } else {
                    this._graphConfiguration.axes = this._config.axes;
                }
                this._graphConfiguration.chartAxisPadding = new Padding(this._graphConfiguration.chartAxisPadding.left, this._graphConfiguration.chartAxisPadding.top, this._graphConfiguration.chartAxisPadding.right, this._graphConfiguration.chartAxisPadding.bottom);
                if(this._config.xRange) {
                    this._graphConfiguration.gridX = this._config.xRange;
                }
                this._graphConfiguration.graphScaleWidthLeft = typeof this._config.leftPadding !== "undefined" ? this._config.leftPadding : this._graphConfiguration.graphScaleWidthLeft;
                this._graphConfiguration.graphScaleWidthRight = typeof this._config.rightPadding !== "undefined" ? this._config.rightPadding : this._graphConfiguration.graphScaleWidthRight;
                if(Plugin.Resources.getString("AFG_Title")) {
                    this._graphConfiguration.header.titleConfig.text = Plugin.Resources.getString("AFG_Title");
                }
                if(Plugin.Resources.getString("AFG_Unit")) {
                    this._graphConfiguration.header.titleConfig.unit = Plugin.Resources.getString("AFG_Unit");
                }
                this._graphConfiguration.graphHeight = typeof this._config.height !== "undefined" ? this._config.height : this._graphConfiguration.graphHeight;
                this._hideCursor = this.hideCursor.bind(this);
                this._graphConfiguration.leftScaleBorder = typeof this._config.leftScaleBorder !== "undefined" ? this._config.leftScaleBorder : this._graphConfiguration.leftScaleBorder;
                this._graphConfiguration.rightScaleBorder = typeof this._config.rightScaleBorder !== "undefined" ? this._config.rightScaleBorder : this._graphConfiguration.rightScaleBorder;
                this._isCursorEnabled = typeof this._config.isCursorEnabled !== "undefined" ? this._config.isCursorEnabled : false;
                this._keyPress = this.onKeyPress.bind(this);
            };
            AnimationFrameGraph.prototype.onResize = function (evt) {
                if(this._isExpanded && this.calculateChartSize()) {
                    this.removeFocus();
                    this.render();
                }
            };
            AnimationFrameGraph.prototype.calculateChartSize = function () {
                var isCalculated = false;
                if(this._container.offsetWidth > 0 && this._container.offsetHeight > 0) {
                    var canvasWidth = this._container.offsetWidth - this._graphConfiguration.graphScaleWidthLeft - this._graphConfiguration.graphScaleWidthRight - this._graphConfiguration.leftScaleBorder - this._graphConfiguration.rightScaleBorder;
                    var canvasHeight = this._graphConfiguration.graphHeight;
                    this._canvas.height = canvasHeight;
                    this._canvas.width = canvasWidth;
                    this._graphConfiguration.chartWidth = canvasWidth - this._graphConfiguration.chartAxisPadding.right;
                    this._graphConfiguration.chartHeight = canvasHeight - this._graphConfiguration.chartAxisPadding.bottom;
                    isCalculated = true;
                }
                return isCalculated;
            };
            AnimationFrameGraph.prototype.clearSelection = function (e) {
                if(this._dragDirection !== VisualProfiler.DragDirection.none) {
                    return;
                }
                this._selectionStartXPercent = 0;
                this._selectionWidthPercent = 100;
                this.drawSelection();
                this.raiseSelectionChanged();
            };
            AnimationFrameGraph.prototype.convertToChartAreaPercentage = function (event) {
                var rect = this._canvas.getBoundingClientRect();
                var mouseX = event.clientX - rect.left;
                return (mouseX) / this._canvasDiv.clientWidth * 100;
            };
            AnimationFrameGraph.prototype.drag = function (event) {
                var xPercent = this.convertToChartAreaPercentage(event);
                this._dragDirection = AnimationFrameGraph.getDragDirection(xPercent - this._initialSelectionStartXPercent);
                this._selectionWidthPercent = Math.abs(xPercent - this._initialSelectionStartXPercent);
                if(this._dragDirection === VisualProfiler.DragDirection.left) {
                    this._selectionStartXPercent = this._initialSelectionStartXPercent - this._selectionWidthPercent;
                }
                this.drawSelection();
                this.raiseSelectionChanged();
            };
            AnimationFrameGraph.prototype.drawChartBackground = function () {
                this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                this._context.fillRect(0, 0, this._canvas.width, this._canvas.height);
            };
            AnimationFrameGraph.prototype.drawSelection = function () {
                if(!this._leftUnselectedRegion || !this._rightUnselectedRegion) {
                    this.initializeUnselectedRegions(this._selectionStartXPercent);
                }
                this._leftUnselectedRegion.style.width = this._selectionStartXPercent + "%";
                this._rightUnselectedRegion.style.left = (this._selectionStartXPercent + this._selectionWidthPercent) + "%";
                this._rightUnselectedRegion.style.width = (100 - (this._selectionStartXPercent + this._selectionWidthPercent)) + "%";
            };
            AnimationFrameGraph.prototype.ensureMinSelectionWidth = function () {
                if(typeof this._selectionStartXPercent === "undefined") {
                    return;
                }
                if(Math.abs(this._selectionWidthPercent * this._canvasDiv.clientWidth / 100) < this._config.minSelectionWidthInPixels) {
                    this._selectionWidthPercent = this._config.minSelectionWidthInPixels / this._canvasDiv.clientWidth * 100;
                    switch(this._dragDirection) {
                        case VisualProfiler.DragDirection.right: {
                            if(this._selectionStartXPercent + this._selectionWidthPercent > 100) {
                                this._selectionStartXPercent = 100 - this._selectionWidthPercent;
                            }
                            break;

                        }
                        case VisualProfiler.DragDirection.left: {
                            if(this._selectionStartXPercent - this._selectionWidthPercent < 0) {
                                this._selectionStartXPercent = 0;
                            } else {
                                if(this._selectionStartXPercent + this._selectionWidthPercent > 100) {
                                    this._selectionStartXPercent = 100 - this._selectionWidthPercent;
                                } else {
                                    this._selectionStartXPercent = this._initialSelectionStartXPercent - this._selectionWidthPercent;
                                }
                            }
                            break;

                        }
                        default: {
                            if(this._selectionStartXPercent + (this._selectionWidthPercent / 2) > 100) {
                                this._selectionStartXPercent = 100 - this._selectionWidthPercent;
                            } else {
                                if(this._selectionStartXPercent - (this._selectionWidthPercent / 2) < 0) {
                                    this._selectionStartXPercent = 0;
                                } else {
                                    this._selectionStartXPercent = this._selectionStartXPercent - (this._selectionWidthPercent / 2);
                                }
                            }
                            break;

                        }
                    }
                }
            };
            AnimationFrameGraph.prototype.convertChartAreaPercentToDataValue = function (percent) {
                return Math.round(percent * this.xWidth / 100) + this._config.xRange.min;
            };
            AnimationFrameGraph.prototype.raiseSelectionChanged = function () {
                if(!this._config.selectionCallback) {
                    return;
                }
                var minValue = this.convertChartAreaPercentToDataValue(this._selectionStartXPercent);
                var maxValue = this.convertChartAreaPercentToDataValue(this._selectionStartXPercent + this._selectionWidthPercent);
                var begin = VisualProfiler.TimeStamp.fromNanoseconds(minValue);
                var end = VisualProfiler.TimeStamp.fromNanoseconds(maxValue);
                var selection = new VisualProfiler.TimeSpan(begin, end);
                this._config.selectionCallback(selection);
            };
            AnimationFrameGraph.prototype.initializeUnselectedRegions = function (startXPercent) {
                this._leftUnselectedRegion = this.createUnselectedRegion(0, startXPercent);
                this._rightUnselectedRegion = this.createUnselectedRegion(startXPercent, 100 - startXPercent);
                this._canvasDiv.appendChild(this._leftUnselectedRegion);
                this._canvasDiv.appendChild(this._rightUnselectedRegion);
            };
            AnimationFrameGraph.prototype.calculateYRange = function (point) {
                if(point) {
                    if(this._graphConfiguration.gridY.min > point.y) {
                        this._graphConfiguration.gridY.min = Math.floor((point.y * this._graphConfiguration.chartAxisIncreaseRatio) / this._graphConfiguration.chartAxisCount) * this._graphConfiguration.chartAxisCount;
                    }
                    if(this._graphConfiguration.gridY.max < point.y + Math.ceil(this._graphConfiguration.gridY.max * this._graphConfiguration.scaleForWhiteSpace)) {
                        this._graphConfiguration.gridY.max = Math.ceil((point.y * this._graphConfiguration.chartAxisIncreaseRatio) / this._graphConfiguration.chartAxisCount) * this._graphConfiguration.chartAxisCount;
                    }
                }
            };
            AnimationFrameGraph.prototype.addSeries = function (legendText, counterId, colorSchemeType) {
                this._graphConfiguration.colorSchemeType = typeof colorSchemeType !== "undefined" ? colorSchemeType : this._graphConfiguration.colorSchemeType;
                var newSeries = new StepLineSeries(this._context, this._graphConfiguration, this._series.length, legendText, counterId);
                this._series.push(newSeries);
                var filledLines = 0;
                for(var i = 0; i < this._series.length; i++) {
                    filledLines++;
                    if(filledLines >= 2) {
                        this._graphConfiguration.chartDrawFill = false;
                        break;
                    }
                }
            };
            AnimationFrameGraph.prototype.addVerticalRulerLines = function () {
                if(this._config.getVerticalRulerLinePositions) {
                    var positions = this._config.getVerticalRulerLinePositions(this._canvas.width);
                    for(var i = 0; positions && i < positions.length; ++i) {
                        this.createVerticalRulerLine(positions[i]);
                    }
                }
            };
            AnimationFrameGraph.prototype.createVerticalRulerLine = function (position) {
                var x = this._graphConfiguration.chartWidth * position / 100;
                x = Math.round(x) + 0.5;
                this._context.save();
                this._context.beginPath();
                this._context.strokeStyle = Plugin.Theme.getValue("vs-combo-box-popup-border-color");
                this._context.lineWidth = 1;
                this._context.moveTo(x, 0);
                this._context.lineTo(x, this._graphConfiguration.chartHeight);
                this._context.closePath();
                this._context.stroke();
                this._context.restore();
            };
            AnimationFrameGraph.prototype.getMaxPoint = function (points) {
                var max;
                if(points) {
                    if(points.length > 0) {
                        max = points[0];
                    }
                    for(var i = 1; i < points.length; i++) {
                        if(points[i].y > max.y) {
                            max = points[i];
                        }
                    }
                }
                return max;
            };
            AnimationFrameGraph.prototype.drawToolTip = function (points, mousePosition) {
                if(!points || points.length === 0) {
                    return;
                }
                var pointsWithMessages = [];
                for(var i = 0; i < points.length; i++) {
                    if(points[i].data !== null && points[i].data !== undefined && points[i].data !== "undefined" && points[i].data !== "") {
                        pointsWithMessages.push(points[i].data);
                    }
                }
                if(pointsWithMessages.length > 0) {
                    var message = "";
                    var separator = "<div class='line-separator'></div>";
                    var count = pointsWithMessages.length - 1;
                    while(count >= 0) {
                        message = pointsWithMessages[count] + message;
                        if(count > 0) {
                            message = separator + message;
                        }
                        count--;
                    }
                    var config = {
                        content: message,
                        delay: 0
                    };
                    Plugin.VS.Tooltip.show(config);
                }
            };
            AnimationFrameGraph.prototype.getMousePosition = function (evt) {
                var rect = this._canvas.getBoundingClientRect();
                var root = document.documentElement;
                var mouseX = evt.clientX - rect.left;
                var mouseY = evt.clientY - rect.top;
                return new Point2d(mouseX, mouseY, null, null, null, evt.screenX, evt.screenY);
            };
            AnimationFrameGraph.prototype.onMouseOver = function (evt) {
                var mousePos = this.getMousePosition(evt);
                if((!this._oldMousePosition || this._oldMousePosition.x !== mousePos.x) && mousePos.x <= this._graphConfiguration.chartWidth - 2 && mousePos.x >= this._graphConfiguration.chartAxisPadding.left && mousePos.y <= this._graphConfiguration.chartHeight && mousePos.y >= this._graphConfiguration.chartAxisPadding.top) {
                    Plugin.VS.Tooltip.dismiss();
                    var foundPoints = this.getPointsBelowMousePointer(mousePos);
                    if(this._isCursorEnabled) {
                        this.drawCursor(mousePos);
                    }
                    if(foundPoints && foundPoints.length !== 0) {
                        this.drawToolTip(foundPoints, mousePos);
                    }
                }
                this._oldMousePosition = mousePos;
            };
            AnimationFrameGraph.prototype.hideCursor = function () {
                if(this._isCursorEnabled) {
                    var mdiv = AnimationFrameGraph._cursor;
                    if(!mdiv) {
                        mdiv = AnimationFrameGraph._cursor = document.getElementById("mDiv");
                    }
                    if(mdiv) {
                        mdiv.style.visibility = "hidden";
                    }
                }
                Plugin.VS.Tooltip.dismiss();
            };
            AnimationFrameGraph.prototype.onMouseOut = function (evt) {
                var mousePos = this.getMousePosition(evt);
                if(mousePos.x > this._graphConfiguration.chartWidth - 2 || mousePos.x < this._graphConfiguration.chartAxisPadding.left || mousePos.y > this._graphConfiguration.chartHeight || mousePos.y < this._graphConfiguration.chartAxisPadding.top) {
                    this.hideCursor();
                }
            };
            AnimationFrameGraph.prototype.drawCursor = function (point) {
                var mdiv = AnimationFrameGraph._cursor;
                if(!mdiv) {
                    mdiv = AnimationFrameGraph._cursor = document.getElementById("mDiv");
                }
                if(!mdiv) {
                    mdiv = document.createElement("div");
                    mdiv.className = "graph-cursor";
                    mdiv.id = "mDiv";
                    mdiv.style.left = point.x.toString() + "px";
                    mdiv.style.top = this._graphConfiguration.chartAxisPadding.top.toString() + "px";
                    mdiv.style.height = (this._canvas.height).toString() + "px";
                    mdiv.style.visibility = "visible";
                    mdiv.style.zIndex = "2";
                    this._canvasDiv.appendChild(mdiv);
                } else {
                    mdiv.style.left = point.x.toString() + "px";
                    mdiv.style.top = this._graphConfiguration.chartAxisPadding.top.toString() + "px";
                    mdiv.style.visibility = "visible";
                }
            };
            AnimationFrameGraph.prototype.getPointsBelowMousePointer = function (point) {
                var foundPoints = [];
                for(var i = 0; i < this._series.length; i++) {
                    var seriesData = this._series[i].renderedPoints;
                    if(seriesData && seriesData.length > 0) {
                        for(var j = 1; j < seriesData.length; j++) {
                            var previousPoint = seriesData[j - 1];
                            var renderedPoint = seriesData[j];
                            if(typeof renderedPoint.xPx !== "undefined" && typeof previousPoint.xPx !== "undefined" && point.x < renderedPoint.xPx && point.x >= previousPoint.xPx) {
                                foundPoints.push(previousPoint);
                                break;
                            }
                        }
                    }
                }
                return foundPoints;
            };
            AnimationFrameGraph.prototype.startSelection = function (event) {
                this.hideCursor();
                this.removeFocus();
                this._initialSelectionStartXPercent = this._selectionStartXPercent = this.convertToChartAreaPercentage(event);
                this._selectionWidthPercent = 0;
                this._dragDirection = AnimationFrameGraph.getDragDirection(this._selectionWidthPercent);
                this._mouseupHandler = this.stopSelection.bind(this);
                this._canvasDiv.addEventListener("mousemove", this._dragHandler);
                this._canvasDiv.addEventListener("mouseup", this._mouseupHandler);
                this._canvasDiv.setCapture(true);
                this.drawSelection();
            };
            AnimationFrameGraph.prototype.stopSelection = function (event) {
                this._canvasDiv.removeEventListener("mousemove", this._dragHandler);
                this._canvasDiv.removeEventListener("mouseup", this._mouseupHandler);
                this._canvasDiv.releaseCapture();
                this.ensureMinSelectionWidth();
                this.drawSelection();
                this._dragDirection = VisualProfiler.DragDirection.none;
                this.createPointIndexInfoList();
                this.raiseSelectionChanged();
            };
            AnimationFrameGraph.prototype.createUnselectedRegion = function (xPercent, widthPercent) {
                var rect = document.createElement("div");
                rect.style.height = this._canvasDiv.offsetHeight + "px";
                rect.style.width = widthPercent + "%";
                rect.style.left = xPercent + "%";
                rect.style.top = "0px";
                rect.className = "unselected";
                rect.addEventListener("mouseover", this._hideCursor);
                return rect;
            };
            AnimationFrameGraph.getDragDirection = function getDragDirection(selectionWidth) {
                if(selectionWidth > 0) {
                    return VisualProfiler.DragDirection.right;
                } else {
                    if(selectionWidth < 0) {
                        return VisualProfiler.DragDirection.left;
                    }
                }
                return VisualProfiler.DragDirection.none;
            }
            AnimationFrameGraph.prototype.onKeyPress = function (evt) {
                if(this._series && (KeyCodes.ARROW_LEFT === evt.keyCode || KeyCodes.ARROW_RIGHT === evt.keyCode)) {
                    if(KeyCodes.ARROW_LEFT === evt.keyCode) {
                        this._index--;
                        if(this._index < 0) {
                            this._index = this._tabFocusPointList.length - 1;
                        }
                    } else {
                        if(KeyCodes.ARROW_RIGHT === evt.keyCode) {
                            this._index++;
                            if(this._index >= this._tabFocusPointList.length) {
                                this._index = 0;
                            }
                        }
                    }
                    this.focusPoint(this._tabFocusPointList[this._index]);
                } else {
                    if(KeyCodes.TAB === evt.keyCode) {
                        this.removeFocus();
                    }
                }
            };
            AnimationFrameGraph.prototype.focusPoint = function (pointInfo) {
                var isMoved = false;
                var series = this._series[this._tabFocusPointList[this._index].seriesIndex];
                if(series && series.renderedPoints) {
                    var points = series.renderedPoints;
                    var previousPoint = points[this._tabFocusPointList[this._index].pointIndex - 1];
                    var nextPoint = points[this._tabFocusPointList[this._index].pointIndex];
                    var divStart;
                    var divEnd;
                    if(previousPoint && nextPoint) {
                        divStart = previousPoint.x - this._selectedTimeRange.min > 0 ? previousPoint.x : this._selectedTimeRange.min;
                        divEnd = this._selectedTimeRange.max - nextPoint.x > 0 ? nextPoint.x : this._selectedTimeRange.max;
                    }
                    if(typeof divEnd !== "undefined" && typeof divStart !== "undefined") {
                        var divs = this._canvasDiv.getElementsByClassName("graph-focus");
                        while(divs.length > 0) {
                            this._canvasDiv.removeChild(divs[0]);
                        }
                        var focusDiv = document.createElement("div");
                        focusDiv.classList.add("graph-focus");
                        focusDiv.style.width = ((divEnd - divStart) / this._graphConfiguration.gridX.range * this._canvasDiv.clientWidth) + "px";
                        focusDiv.style.left = ((divStart - this._graphConfiguration.gridX.min) / this._graphConfiguration.gridX.range * this._canvasDiv.clientWidth) - 1 + "px";
                        focusDiv.style.top = previousPoint.yPx - 1 + "px";
                        var ariaLabel = Plugin.Resources.getString("VisualThroughputStepAriaLabel", previousPoint.y.toString(), Graph.RulerUtilities.formatTime(VisualProfiler.TimeStamp.fromNanoseconds(previousPoint.x), Graph.UnitFormat.fullName));
                        focusDiv.setAttribute("aria-label", ariaLabel);
                        this._canvasDiv.appendChild(focusDiv);
                        isMoved = true;
                    }
                }
                return isMoved;
            };
            AnimationFrameGraph.prototype.createPointIndexInfoList = function () {
                var startX = this.convertChartAreaPercentToDataValue(this._selectionStartXPercent);
                var endX = this.convertChartAreaPercentToDataValue(this._selectionStartXPercent + this._selectionWidthPercent);
                this._selectedTimeRange = new MinMax(startX, endX);
                this._tabFocusPointList = this.getPointIndexInfoList(startX, endX);
                this._index = -1;
            };
            AnimationFrameGraph.prototype.getPointIndexInfoList = function (start, end) {
                var lists = [];
                if(this._series) {
                    for(var i = 0; i < this._series.length; i++) {
                        var series = this._series[i];
                        var seriesPoints = series.renderedPoints;
                        var list = [];
                        for(var j = 0; j < seriesPoints.length; j++) {
                            var seriesPoint = seriesPoints[j];
                            var required = seriesPoint.x > start && seriesPoint.x <= end;
                            if(!required) {
                                if(seriesPoint.x > end && j > 0 && seriesPoints[j - 1].x < end) {
                                    required = true;
                                }
                            }
                            if(required) {
                                list.push({
                                    seriesIndex: i,
                                    pointIndex: j
                                });
                            }
                        }
                        lists.push(list);
                    }
                }
                return this.sortPointIndexInfoList(lists);
            };
            AnimationFrameGraph.prototype.sortPointIndexInfoList = function (lists) {
                var finalList = [];
                if(lists) {
                    while(!this.areAllListEmpty(lists)) {
                        var smallest = null;
                        var index = null;
                        for(var i = 0; i < lists.length; i++) {
                            var list = lists[i];
                            if(list.length > 0) {
                                var smallestInList = list[0];
                                if(smallestInList && (!smallest || this._series[smallestInList.seriesIndex].renderedPoints[smallestInList.pointIndex].x < this._series[smallest.seriesIndex].renderedPoints[smallest.pointIndex].x)) {
                                    smallest = smallestInList;
                                    index = i;
                                }
                            }
                        }
                        if(smallest) {
                            finalList.push(lists[index].shift());
                        }
                    }
                }
                return finalList;
            };
            AnimationFrameGraph.prototype.areAllListEmpty = function (lists) {
                var allListEmpty = true;
                if(lists) {
                    for(var i = 0; i < lists.length; i++) {
                        if(lists[i].length > 0) {
                            allListEmpty = false;
                            break;
                        }
                    }
                }
                return allListEmpty;
            };
            AnimationFrameGraph.prototype.removeFocus = function () {
                var divs = this._canvasDiv.getElementsByClassName("graph-focus");
                while(divs.length > 0) {
                    this._canvasDiv.removeChild(divs[0]);
                }
                this._index = -1;
            };
            return AnimationFrameGraph;
        })();
        Graph.AnimationFrameGraph = AnimationFrameGraph;        
        var ChartSeries = (function () {
            function ChartSeries(context, index, legendText, counterId, config) {
                this.colorSchemes = [
                    new ChartColorScheme("rgb(198, 198, 198)", "rgba(198, 198, 198, 0.75)"), 
                    new ChartColorScheme("rgb(83, 134, 191)", "rgba(83, 134, 191, 0.50)"), 
                    new ChartColorScheme("rgb(167, 148, 50)", "rgba(167, 148, 50, 0.25)"), 
                    new ChartColorScheme("rgb(158, 202, 0)", "rgba(158, 202, 0, 0.65)"), 
                    
                ];
                this.index = index;
                this.legendText = legendText;
                this.data = [];
                this.colorScheme = config.colorSchemeType < this.colorSchemes.length ? this.colorSchemes[config.colorSchemeType] : this.colorSchemes[3];
                this.counterId = counterId;
                this.graphConfiguration = config;
                this.context = context;
            }
            ChartSeries.prototype.getPointsToRender = function (chartRect) {
                var renderedPoints = new Array();
                var seriesPoints = this.data;
                var seriesIndex = this.index;
                if(seriesPoints.length > 0) {
                    var startIndex = 0;
                    var endIndex = seriesPoints.length - 1;
                    var startX = this.graphConfiguration.gridX.min;
                    var rangeX = this.graphConfiguration.gridX.range;
                    var endX = this.graphConfiguration.gridX.max;
                    var scaleX = chartRect.width / rangeX;
                    var x = 0;
                    var y = 0;
                    var startPoint = seriesPoints[startIndex];
                    var hasMovedToStart = false;
                    for(var i = startIndex; i <= endIndex; i++) {
                        var point = seriesPoints[i];
                        var pointInView = (point.x >= startX && point.x <= endX);
                        var requiresDrawing = pointInView;
                        point.isInView = pointInView;
                        point.seriesIndex = seriesIndex;
                        if(!requiresDrawing) {
                            if((point.x < startX && i < endIndex && seriesPoints[i + 1].x > startX) || (point.x > endX && i > 0 && seriesPoints[i - 1].x < endX)) {
                                requiresDrawing = true;
                            }
                        }
                        if(requiresDrawing) {
                            x = chartRect.left + (point.x - startX) * scaleX;
                            y = this.getYCoordinate(point.y, chartRect.top, chartRect.height);
                            point.xPx = Math.floor(x);
                            point.yPx = Math.floor(y);
                            renderedPoints.push(point);
                        }
                    }
                    if(renderedPoints && renderedPoints.length > 0 && renderedPoints[0].x === seriesPoints[0].x && renderedPoints[0].y === seriesPoints[0].y && renderedPoints[0].x >= startX) {
                        var arr = [];
                        arr.push(new Point2d(seriesPoints[startIndex].x, this.graphConfiguration.gridY.min, seriesPoints[startIndex].x.toString(), false, seriesIndex, 0, 0, chartRect.left + (seriesPoints[startIndex].x - startX) * scaleX, this.getYCoordinate(this.graphConfiguration.gridY.min, chartRect.top, chartRect.height)));
                        renderedPoints = arr.concat(renderedPoints);
                    }
                }
                return renderedPoints;
            };
            ChartSeries.prototype.getYCoordinate = function (y, top, height) {
                return top + ((this.graphConfiguration.gridY.max - y) / this.graphConfiguration.gridY.range) * height;
            };
            return ChartSeries;
        })();
        Graph.ChartSeries = ChartSeries;        
        var StepLineSeries = (function (_super) {
            __extends(StepLineSeries, _super);
            function StepLineSeries(context, config, index, legendText, counterId) {
                        _super.call(this, context, index, legendText, counterId, config);
            }
            StepLineSeries.prototype.drawLineFill = function (renderedPoints) {
                if(renderedPoints.length > 0) {
                    var startIndex = 0;
                    var endIndex = renderedPoints.length - 1;
                    var startX = this.graphConfiguration.gridX.min;
                    var rangeX = this.graphConfiguration.gridX.range;
                    var endX = this.graphConfiguration.gridX.max;
                    var scaleX = this.graphConfiguration.chartRect.width / rangeX;
                    var x = 0;
                    var y = 0;
                    var hasMovedToStart = false;
                    this.context.save();
                    this.context.beginPath();
                    var previousPoint = renderedPoints[0];
                    this.context.beginPath();
                    this.context.fillStyle = this.colorScheme.lineFillColor;
                    this.context.strokeStyle = this.colorScheme.lineColor;
                    for(var i = startIndex; i <= endIndex; i++) {
                        var point = renderedPoints[i];
                        if(!hasMovedToStart) {
                            this.context.moveTo(point.xPx, point.yPx);
                            hasMovedToStart = true;
                        } else {
                            this.context.lineTo(point.xPx, previousPoint.yPx);
                            this.context.lineTo(point.xPx, point.yPx);
                        }
                        previousPoint = point;
                    }
                    if(this.graphConfiguration.chartDrawFill) {
                        var y = this.getYCoordinate(0, this.graphConfiguration.chartRect.top, this.graphConfiguration.chartRect.height);
                        this.context.lineTo(point.xPx, y);
                        this.context.lineTo(this.graphConfiguration.chartAxisPadding.left, y);
                        this.context.closePath();
                        this.context.fill();
                    } else {
                        this.context.stroke();
                    }
                    this.context.restore();
                }
            };
            StepLineSeries.prototype.drawLineSeries = function (renderedPoints) {
                if(renderedPoints.length > 0) {
                    this.context.save();
                    if(this.graphConfiguration.chartDrawFill) {
                        this.context.beginPath();
                        this.context.lineWidth = (this.graphConfiguration.chartRect.height < 100 ? 1 : 2);
                        this.context.fillStyle = this.colorScheme.lineFillColor;
                        this.context.strokeStyle = this.colorScheme.lineColor;
                        var previousPoint = null;
                        for(var i = 0; i < renderedPoints.length; i++) {
                            var rp = renderedPoints[i];
                            if(i === 0) {
                                this.context.moveTo(renderedPoints[i].xPx, renderedPoints[i].yPx - 0.5);
                            } else {
                                var yPos = this.graphConfiguration.chartHeight - 0.5;
                                if(previousPoint && previousPoint.yPx) {
                                    yPos = Math.round(previousPoint.yPx) - 0.5;
                                }
                                this.context.lineTo(rp.xPx, yPos);
                                this.context.lineTo(rp.xPx, Math.round(rp.yPx) - 0.5);
                            }
                            previousPoint = rp;
                        }
                        this.context.stroke();
                    }
                    this.context.restore();
                }
            };
            StepLineSeries.prototype.render = function () {
                this.renderedPoints = this.getPointsToRender(this.graphConfiguration.chartRect);
                if(this.renderedPoints && this.renderedPoints.length > 0) {
                    this.drawLineFill(this.renderedPoints);
                    this.drawLineSeries(this.renderedPoints);
                }
            };
            return StepLineSeries;
        })(ChartSeries);
        Graph.StepLineSeries = StepLineSeries;        
        var Axes = (function () {
            function Axes(config) {
                this._graphConfiguration = config;
                this._graphAxes = document.getElementsByClassName("graph-canvas-div")[0];
                if(!this._graphAxes) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
            }
            Axes.prototype.drawAxisLine = function (position, value) {
                var axisLine = document.createElement("div");
                axisLine.className = "graph-axis-line";
                this._graphAxes.appendChild(axisLine);
                var top = 0;
                switch(position) {
                    case AxisPositionType.top: {
                        top = 0;
                        break;

                    }
                    case AxisPositionType.middle: {
                        var y = Math.floor(this._graphConfiguration.chartAxisPadding.top + ((this._graphConfiguration.gridY.max - value) / this._graphConfiguration.gridY.range) * (this._graphConfiguration.chartHeight - this._graphConfiguration.chartAxisPadding.top));
                        top = y - (axisLine.offsetHeight / 2);
                        break;

                    }
                    case AxisPositionType.bottom: {
                        var y = Math.floor(this._graphConfiguration.chartAxisPadding.top + this._graphConfiguration.chartHeight);
                        top = y - axisLine.offsetHeight;
                        break;

                    }
                }
                axisLine.style.top = top + "px";
            };
            Axes.prototype.render = function () {
                var children = this._graphAxes.getElementsByClassName("graph-axis-line");
                if(children) {
                    while(children.length > 0) {
                        this._graphAxes.removeChild(children[children.length - 1]);
                    }
                }
                if(this._graphConfiguration.axes && this._graphConfiguration.axes.length > 0) {
                    for(var i = 0; i < this._graphConfiguration.axes.length; i++) {
                        var axis = this._graphConfiguration.axes[i];
                        this.drawAxisLine(AxisPositionType.middle, axis.value);
                    }
                } else {
                    this.drawAxisLine(AxisPositionType.top, this._graphConfiguration.gridY.max);
                    this.drawAxisLine(AxisPositionType.bottom, this._graphConfiguration.gridY.min);
                }
            };
            Object.defineProperty(Axes.prototype, "height", {
                set: function (height) {
                    this._graphAxes.style.height = height + "px";
                },
                enumerable: true,
                configurable: true
            });
            return Axes;
        })();
        Graph.Axes = Axes;        
        var Scale = (function () {
            function Scale(type, config) {
                this._graphConfiguration = config;
                this._type = type;
                this._width = type === ScaleType.Left ? this._graphConfiguration.graphScaleWidthLeft : this._graphConfiguration.graphScaleWidthRight;
                this._graphScale = document.createElement("div");
                this._graphScale.className = type === ScaleType.Left ? "graph-scale-left" : "graph-scale-right";
                this._graphScale.style.width = this._width + "px";
                if(type === ScaleType.Left) {
                    this._graphScale.style.borderRightWidth = this._graphConfiguration.leftScaleBorder + "px";
                } else {
                    this._graphScale.style.borderLeftWidth = this._graphConfiguration.rightScaleBorder + "px";
                }
                var container = document.getElementById(this._graphConfiguration.graphContainerId);
                container.appendChild(this._graphScale);
            }
            Scale.prototype.drawAxisValue = function (position, value) {
                var axisDiv = document.createElement("div");
                axisDiv.className = this._type === ScaleType.Left ? "graph-axis-left" : "graph-axis-right";
                axisDiv.innerHTML = value.toString();
                this._graphScale.appendChild(axisDiv);
                var top = 0;
                switch(position) {
                    case AxisPositionType.top: {
                        top = 0;
                        break;

                    }
                    case AxisPositionType.middle: {
                        var y = Math.floor(this._graphConfiguration.chartAxisPadding.top + ((this._graphConfiguration.gridY.max - value) / this._graphConfiguration.gridY.range) * (this._graphConfiguration.chartHeight - this._graphConfiguration.chartAxisPadding.top));
                        top = y - (axisDiv.offsetHeight / 2);
                        break;

                    }
                    case AxisPositionType.bottom: {
                        var y = Math.floor(this._graphConfiguration.chartAxisPadding.top + this._graphConfiguration.chartHeight);
                        top = y - axisDiv.offsetHeight;
                        break;

                    }
                }
                axisDiv.style.top = top + "px";
            };
            Scale.prototype.render = function () {
                while(this._graphScale.childNodes.length > 0) {
                    this._graphScale.removeChild(this._graphScale.firstChild);
                }
                if(this._graphConfiguration.axes && this._graphConfiguration.axes.length > 0) {
                    for(var i = 0; i < this._graphConfiguration.axes.length; i++) {
                        var axis = this._graphConfiguration.axes[i];
                        this.drawAxisValue(AxisPositionType.middle, axis.value);
                    }
                } else {
                    this.drawAxisValue(AxisPositionType.top, this._graphConfiguration.gridY.max);
                    this.drawAxisValue(AxisPositionType.bottom, this._graphConfiguration.gridY.min);
                }
            };
            return Scale;
        })();
        Graph.Scale = Scale;        
        var Legend = (function () {
            function Legend(config) {
                if(!config || !config.containerId) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
                this._config = config;
            }
            Legend.prototype.render = function () {
                for(var i = 0; i < this._config.data.length; i++) {
                    var legendColorDiv = document.createElement("div");
                    legendColorDiv.className = "legend-color-div";
                    var colorDiv = document.createElement("div");
                    colorDiv.className = "legend-color";
                    colorDiv.style.backgroundColor = this._config.data[i].color;
                    legendColorDiv.appendChild(colorDiv);
                    var legendTextDiv = document.createElement("div");
                    legendTextDiv.className = "legend-text-div";
                    legendTextDiv.innerText = this._config.data[i].text;
                    var individualLegendBox = document.createElement("div");
                    individualLegendBox.className = "legend-div";
                    individualLegendBox.appendChild(legendColorDiv);
                    individualLegendBox.appendChild(legendTextDiv);
                    var container = document.getElementById(this._config.containerId);
                    container.appendChild(individualLegendBox);
                }
            };
            return Legend;
        })();
        Graph.Legend = Legend;        
        var Title = (function () {
            function Title(config) {
                this._callbackArr = [];
                this._focused = false;
                this._config = config;
            }
            Title.prototype.onTabFocus = function () {
                this._focused = true;
            };
            Title.prototype.onTabBlur = function () {
                this._focused = false;
            };
            Title.prototype.onKeyPress = function (e) {
                if(this._focused && KeyCodes.ENTER === e.keyCode) {
                    this.mouseClick();
                }
            };
            Title.prototype.mouseClick = function () {
                this._config.isExpanded = !this._config.isExpanded;
                this.setAraiLabelAndStylesheet();
                for(var i = 0; i < this._callbackArr.length; i++) {
                    this._callbackArr[i](this._config.isExpanded);
                }
            };
            Title.prototype.setAraiLabelAndStylesheet = function () {
                if(this._config.isExpanded === null || this._config.isExpanded === undefined || this._config.isExpanded) {
                    this._titleCollapseExpandButton.className = "title-expand-button";
                    this._titleCollapseExpandButton.setAttribute("aria-label", Plugin.Resources.getString("GraphCollapseAriaLabel"));
                    this._titleCollapseExpandButton.setAttribute("aria-expanded", "true");
                } else {
                    this._titleCollapseExpandButton.className = "title-collapse-button";
                    this._titleCollapseExpandButton.setAttribute("aria-label", Plugin.Resources.getString("GraphExpandAriaLabel"));
                    this._titleCollapseExpandButton.setAttribute("aria-expanded", "false");
                }
            };
            Title.prototype.render = function () {
                var titleText = document.createElement("div");
                titleText.className = "title-text";
                var text = "";
                if(this._config.text) {
                    text = this._config.text;
                }
                if(this._config.unit) {
                    text += " (" + this._config.unit + ")";
                }
                titleText.innerText = text;
                this._titleCollapseExpandButton = document.createElement("div");
                this._titleCollapseExpandButton.setAttribute("role", "region");
                this.setAraiLabelAndStylesheet();
                this._titleCollapseExpandButton.addEventListener("click", this.mouseClick.bind(this));
                this._titleCollapseExpandButton.tabIndex = 0;
                this._titleCollapseExpandButton.addEventListener("focus", this.onTabFocus.bind(this));
                this._titleCollapseExpandButton.addEventListener("blur", this.onTabBlur.bind(this));
                this._titleCollapseExpandButton.addEventListener("keypress", this.onKeyPress.bind(this));
                var container = document.getElementById(this._config.containerId);
                container.appendChild(this._titleCollapseExpandButton);
                container.appendChild(titleText);
            };
            Title.prototype.registerCallback = function (callback) {
                this._callbackArr.push(callback);
            };
            return Title;
        })();
        Graph.Title = Title;        
        var Header = (function () {
            function Header(config) {
                var titleContainer = document.createElement("div");
                titleContainer.id = config.titleConfig.containerId;
                titleContainer.className = "title-container";
                var legendContainer = document.createElement("div");
                legendContainer.id = config.legendConfig.containerId;
                legendContainer.className = "legend-container";
                var container = document.getElementById(config.containerId);
                container.appendChild(titleContainer);
                container.appendChild(legendContainer);
                this.title = new Title(config.titleConfig);
                this.legend = new Legend(config.legendConfig);
            }
            Header.prototype.render = function () {
                this.title.render();
                this.legend.render();
            };
            return Header;
        })();
        Graph.Header = Header;        
    })(VisualProfiler.Graph || (VisualProfiler.Graph = {}));
    var Graph = VisualProfiler.Graph;
})(VisualProfiler || (VisualProfiler = {}));
//@ sourceMappingURL=AnimationFrameGraph.js.map

// SIG // Begin signature block
// SIG // MIIaowYJKoZIhvcNAQcCoIIalDCCGpACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFOjfETJLC/ui
// SIG // eBz/xW5qgKPwSW8koIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSWMIIEkgIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIG4MBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBREOum7Of7Jx0frhRIXy5Ao
// SIG // nULsgjBYBgorBgEEAYI3AgEMMUowSKAugCwAQQBuAGkA
// SIG // bQBhAHQAaQBvAG4ARgByAGEAbQBlAEcAcgBhAHAAaAAu
// SIG // AGoAc6EWgBRodHRwOi8vbWljcm9zb2Z0LmNvbTANBgkq
// SIG // hkiG9w0BAQEFAASCAQDU9UvHzNHRFyzCFT5Ax9+ONUy1
// SIG // cpd6XyBsTTThhhHBklpRjlNJtj8JikjJxX+FsVbb0Pna
// SIG // fhvlGtQiKrdXFbnBM8bNbJy/hBVf7HA7PP/gx8odOeRP
// SIG // L5GRs8SoNEmzSeCWxvVqXee2z/HQes4hOva7YATC0IlA
// SIG // uX8xiLLtsLOTYGvvdn877kiJ0+YShlMu4XeVUwvXKa0y
// SIG // F5aCAZx3J7gyTfvEjwiHIvD5V6TNcBF/yvyMHAMixmDt
// SIG // j8wxHSHolrj3QE28XNj0QoG7HCRfjDnPPzeDkzx32dXZ
// SIG // bcRje+b34zrGx+7VhKoPcq5k0VLQ3CX1GLxwIiRYIxYh
// SIG // yusQQJG5oYICHzCCAhsGCSqGSIb3DQEJBjGCAgwwggII
// SIG // AgEBMIGFMHcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xITAfBgNV
// SIG // BAMTGE1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQQIKYQKS
// SIG // SgAAAAAAIDAJBgUrDgMCGgUAoF0wGAYJKoZIhvcNAQkD
// SIG // MQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUxDxcNMTMw
// SIG // MzE1MDYzNDAyWjAjBgkqhkiG9w0BCQQxFgQUjviHbIMz
// SIG // z/bQZ89JBp70T7cIzZswDQYJKoZIhvcNAQEFBQAEggEA
// SIG // bQ/Es113j3Q+40D4Fkaq0Vs5pDI6eio23HGChZl9+769
// SIG // YcEU7VP1ybPxYOKyvPy962gDcdlTxEHFuR2+U625pn7p
// SIG // vbOIYSeErsi/KB6a1yQ+lgXwYI/tfnfyK5We5elS9MRh
// SIG // dXoXwsD3XYUMSF/h3yeLEQIvdiyLsaNpt+AXNwfJhpMK
// SIG // fbVYPGLWU++xb/RLLwKMHaY9r269Fhvsqo1LxZb2oyCF
// SIG // nQcx15xvgMjTij3f3f4Iw5bi9Jf3q8QwQnRy2fFflDnm
// SIG // 9Y5cS6pTSWF0ZLmZKO617tFQLILfmTFqJW1P0/YVVjYd
// SIG // 3gcVpY+xPPBpfJnmSRnKL3P4amp7e8K45w==
// SIG // End signature block
